<?php
/**
 * Booster for WooCommerce - Module - Product Availability by Date
 *
 * @version 7.1.6
 * @since  1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WCJ_Product_By_Date' ) ) :
	/**
	 * WCJ_Product_By_Date.
	 */
	class WCJ_Product_By_Date extends WCJ_Module {

		public $day_now;
		public $month_now;
		public $meta_box_validate_value;

		public function __construct() {

			$this->id         = 'product_by_date';
			$this->short_desc = __( 'Product Availability by Date', 'woocommerce-jetpack' );
			$this->desc       = __( 'Set product availability by date (Custom frontend messages available in Plus).', 'woocommerce-jetpack' );
			$this->desc_pro   = __( 'Set product availability by date.', 'woocommerce-jetpack' );
			$this->link_slug  = 'woocommerce-product-availability-by-date';
			parent::__construct();
			$this->time_now = wcj_get_timestamp_date_from_gmt();

			if ( $this->is_enabled() ) {
				if ( 'yes' === wcj_get_option( 'wcj_product_by_date_per_product_enabled', 'no' ) ) {
					add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
					add_action( 'save_post_product', array( $this, 'save_meta_box' ), PHP_INT_MAX, 2 );
					add_filter( 'wcj_save_meta_box_value', array( $this, 'save_meta_box_validate_value' ), PHP_INT_MAX, 3 );
					add_action( 'admin_notices', array( $this, 'validate_value_admin_notices' ) );
					$this->meta_box_validate_value = 'wcj_product_by_date_enabled';
				}
				if ( 'yes' === wcj_get_option( 'wcj_product_by_date_per_product_enabled', 'no' ) || 'yes' === wcj_get_option( 'wcj_product_by_date_section_enabled', 'no' ) ) {
					$this->day_now   = intval( gmdate( 'j', $this->time_now ) );
					$this->month_now = intval( gmdate( 'n', $this->time_now ) );
					if ( 'non_purchasable' === wcj_get_option( 'wcj_product_by_date_action', 'non_purchasable' ) ) {
						add_filter( 'woocommerce_is_purchasable', array( $this, 'check_is_purchasable_by_date' ), PHP_INT_MAX, 2 );
					}
					add_action( 'woocommerce_single_product_summary', array( $this, 'maybe_add_unavailable_by_date_message' ), 30 );
					if ( 'yes' === wcj_get_option( 'wcj_product_by_date_show_message_on_shop_enabled', 'no' ) ) {
						add_action( 'woocommerce_shop_loop_item_title', array( $this, 'maybe_add_unavailable_by_date_message' ), 30 );
					}
				}
			}
		}

		public function get_default_date( $i ) {
			$date_defaults = array(
				'1-31',
				'1-29',
				'1-31',
				'1-30',
				'1-31',
				'-',
				'-',
				'-',
				'10-30',
				'1-31',
				'1-30',
				'1-20,26-31',
			);
			return $date_defaults[ $i - 1 ];
		}

		public function maybe_get_direct_date( $product_id ) {
			$direct_date = get_post_meta( $product_id, '_wcj_product_by_date_direct_date', true );
			return (
				'yes' === wcj_get_option( 'wcj_product_by_date_per_product_enabled', 'no' ) &&
				'yes' === get_post_meta( $product_id, '_wcj_product_by_date_enabled', true ) &&
				'' !== ( $direct_date )
			) ? $direct_date : false;
		}

		public function maybe_add_unavailable_by_date_message() {
			$_product = wc_get_product();
			if ( ! $this->check_is_purchasable_by_date( true, $_product ) ) {
				$direct_date = $this->maybe_get_direct_date( wcj_get_product_id_or_variation_parent_id( $_product ) );
				if ( false !== ( $direct_date ) ) {
					$replaceable_values = array(
						'%direct_date%' => $this->convert_to_jalali( $direct_date ),
					);
					$message            = wcj_get_option(
						'wcj_product_by_date_unavailable_message_direct_date',
						'<p style="color:red;">' . __( '%product_title% is not available until %direct_date%.', 'woocommerce-jetpack' ) . '</p>'
					);
				} else {
					$_date              = $this->get_product_availability_this_month( $_product );
					$replaceable_values = array(
						'%date_this_month%' => $_date,
					);
					$message            = ( ( '-' === $_date ) ?
					apply_filters(
						'booster_option',
						'<p style="color:red;">' . __( '%product_title% is not available this month.', 'woocommerce-jetpack' ) . '</p>',
						get_option(
							'wcj_product_by_date_unavailable_message_month_off',
							'<p style="color:red;">' . __( '%product_title% is not available this month.', 'woocommerce-jetpack' ) . '</p>'
						)
					) :
					apply_filters(
						'booster_option',
						'<p style="color:red;">' . __( '%product_title% is available only on %date_this_month% this month.', 'woocommerce-jetpack' ) . '</p>',
						get_option(
							'wcj_product_by_date_unavailable_message',
							'<p style="color:red;">' . __( '%product_title% is available only on %date_this_month% this month.', 'woocommerce-jetpack' ) . '</p>'
						)
					)
					);
				}
				$replaceable_values['%product_title%'] = $_product->get_title();
				echo wp_kses_post( str_replace( array_keys( $replaceable_values ), $replaceable_values, do_shortcode( $message ) ) );
			}
		}

		public function get_product_availability_this_month( $_product ) {
			$product_id = wcj_get_product_id_or_variation_parent_id( $_product );
			if ( 'yes' === wcj_get_option( 'wcj_product_by_date_per_product_enabled', 'no' ) && 'yes' === get_post_meta( $product_id, '_wcj_product_by_date_enabled', true ) ) {
				return get_post_meta( $product_id, '_wcj_product_by_date_' . $this->month_now, true );
			} elseif ( 'yes' === wcj_get_option( 'wcj_product_by_date_section_enabled', 'no' ) ) {
				return wcj_get_option( 'wcj_product_by_date_' . $this->month_now, $this->get_default_date( $this->month_now ) );
			} else {
				return '';
			}
		}

		public function check_is_purchasable_by_date( $purchasable, $_product ) {
			if ( $purchasable ) {
				$direct_date = $this->maybe_get_direct_date( wcj_get_product_id_or_variation_parent_id( $_product ) );
				if ( false !== ( $direct_date ) ) {
					$date = DateTime::createFromFormat( wcj_get_option( 'wcj_product_by_date_direct_date_format', 'm/d/Y' ), $direct_date, wcj_timezone() );
					if ( false === $date ) {
						return false;
					}
					$timestamp = $date->getTimestamp();
					return ( $this->time_now >= $timestamp );
				} else {
					$_date = $this->get_product_availability_this_month( $_product );
					if ( '-' === $_date ) {
						return false;
					} elseif ( '' === $_date ) {
						return true;
					} else {
						return wcj_check_date( $_date, array( 'day_now' => $this->day_now ) );
					}
				}
			}
			return $purchasable;
		}

		/**
		 * Convert Gregorian date to Jalali date
		 *
		 * @param string $gregorian_date Gregorian date in 'm/d/Y' format.
		 * @return string Jalali date in 'Y/m/d' format.
		 */
		private function convert_to_jalali( $gregorian_date ) {
			list( $month, $day, $year ) = explode( '/', $gregorian_date );
			require_once 'jdf.php';
			$jalali_date = gregorian_to_jalali( $year, $month, $day );
			return implode( '/', $jalali_date );
		}
	}

endif;

return new WCJ_Product_By_Date();